/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package hr.algebra.streams;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.text.Collator;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Comparator;
import java.util.IntSummaryStatistics;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.OptionalDouble;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.LongStream;
import java.util.stream.Stream;

/**
 *
 * @author dnlbe
 */
public class Main {

    public static void main(String[] args) {
        basicStreamsDemo();
        terminalsDemo();
        reduceDemo();
        collectDemo();
        parallelDemo();
    }

    // There are many ways to create a stream instance of different sources. Once created, the instance will not
    // modify its source, therefore allowing the creation of multiple instances from a single source.
    private static void basicStreamsDemo() {
        // convert from array[] (...) to list
        List<String> list = Arrays.asList("Daniel", "Daniel", "Milica", "Danica", "Danica", "danica", "đanica");
        list.stream()
                .filter(n -> !n.endsWith("ca")) // intermediate operation
                .forEach(System.out::println); // terminal operation
        System.out.println("Original names - not changed:");
        list.forEach(System.out::println);

        System.out.println("Sorted names, without the duplicates, and first one:");
        // convert from list to array
        var names = list.toArray(String[]::new);
        Collator collator = Collator.getInstance(Locale.forLanguageTag("hr"));
        Arrays.stream(names)
                //.sorted() 
                .distinct() // we should eliminate duplicates first, then sort the rest
                //.sorted() // we still have a problem with uppercase/lowercase letters ordering!
                //.sorted(Comparator.comparing(String::toLowerCase))
                .sorted(Comparator.comparing(String::toLowerCase, collator))
                .skip(1)
                .forEach(System.out::println);

        System.out.println("Uppercase names, length more than 5:");
        Stream.<String>builder()
                .add("Daniel")
                .add("Ivo")
                .add("Pero")
                .add("Milivoj")
                .add("Danica")
                .build()
                //.map(String::toUpperCase)
                .filter(n -> n.length() > 5) // we should filter first, then map ther rest
                .map(String::toUpperCase) // converts (maps) values to their uppercase values
                .forEach(System.out::println);

        // flatMap maps a single element into multiple elements
        System.out.println("Flat map:");
        Stream.of("Daniel Bele", "Milica Krmpotic", "Gojko Mrnjavcevic", "Danica")
                // flatMap expects Function<String, Stream> -> so it converts every value converted into stream -> "Daniel Bele" -> "Daniel", "Bele"
                .flatMap(n -> Arrays.stream(n.split(" ")))
                .forEach(System.out::println);

        System.out.println("Positive, even random numbers:");
        Stream.generate(Random::new)// why do we create so many objects?
                .limit(10)// we need 10 random objects
                .map(Random::nextInt)// we map random objects to random values they create
                .filter(n -> n > 0 && n % 2 == 0)
                .forEach(System.out::println);

        System.out.println("Positive, even random numbers, better:");
        //new Random().ints(10)
        ThreadLocalRandom.current().ints()
                .filter(n -> n > 0 && n % 2 == 0)
                .limit(10) // this is how we accomplish 10 numbers
                .forEach(System.out::println);

        System.out.println("Iterator: Integers from 1 - 5:");
        Stream.iterate(1, n -> ++n) // this does not work (1, n -> n++)!
                .limit(5)
                .forEach(System.out::println);

        System.out.println("Range: Integers from 1 - 5:");
        IntStream
                //.range(1, 6) // endExclusive
                .rangeClosed(1, 5) // endInclusive
                .forEach(System.out::println);

        System.out.println("String as stream of ASCII codes, upper-cased:");
        "Milica Krmpotić".chars()
                //.map(c -> c >= 97 && c <= 122 ? c - 32 : c)
                //.map(c -> Character.isAlphabetic(c) && Character.isUpperCase(c) ? c : Character.toUpperCase(c))
                .map(Character::toUpperCase)
                .forEach(c -> System.out.printf("%c: %d%n", (char)c, c));

        // manual closing
        try (Stream<String> lines = Files.lines(Paths.get("names.txt")/*, StandardCharsets.UTF_8*/)) {
            lines.forEach(System.out::println);
        } catch (Exception e) {
            Logger.getLogger(Main.class.getName()).log(Level.SEVERE, e.getMessage());
        }

        // closed by the system
        try {
            Files.readAllLines(Paths.get("names.txt")).forEach(System.out::println);
        } catch (IOException ex) {
            Logger.getLogger(Main.class.getName()).log(Level.SEVERE, null, ex);
        }

    }

    private static void terminalsDemo() {
        System.out.println("Peek, and average:");
        OptionalDouble average = IntStream.rangeClosed(21, 26)
                .peek(System.out::println) // non terminal operation, provides a peek into the pipeline
                .average();
        if (average.isPresent()) {
            System.out.println("Avg:" + average.getAsDouble());
        }

        List<String> names = Arrays.asList("Ćamil", "Milivoj", "Danica");
        System.out.println("Min:");
        // we do not a return value of Optional
        Collator collator = Collator.getInstance(Locale.forLanguageTag("hr"));
        names.stream()
                //.min(String::compareTo)
                //.min(Comparator.comparing(n -> n, collator))
                .min(Comparator.comparing(Function.identity(), collator))
                .ifPresent(System.out::println);

        System.out.println("Starts with D");
        names.stream()
                .filter(n -> n.startsWith("D"))
                //.findFirst()
                .findAny()
                .ifPresent(System.out::println);

        boolean match = names.stream()
                //.anyMatch(n -> n.contains("ili"));
                //.noneMatch(String::isEmpty);
                .allMatch(n -> n.length() > 3);

        System.out.println(match);

        long count = IntStream.range(1, 100)
                .filter(n -> n % 3 == 0)
                .count();
        System.out.println("Number of numbers dividable with 3 from range 1 - 100: " + count);

    }

    // Reduction stream operations allow us to produce one single result from a sequence of elements, 
    // by applying repeatedly a combining operation to the elements in the sequence.
    // There are three variations of this method, which differ by their signatures and returning types. 
    // They can have the following parameters:
    //      identity    – the initial value for an accumulator or a default value if a stream is empty and 
    //                    there is nothing to accumulate;
    //      accumulator – a function which specifies a logic of aggregation of elements. 
    //                    As accumulator creates  a new value for every step of reducing, the quantity of new values 
    //                    equals to the stream’s size and only the last value is useful. 
    //                    This is not very good for the performance.
    //      combiner    – a function which aggregates results of the accumulator. 
    //                    Combiner is called only in a parallel mode to reduce results of accumulators from different threads.
    private static void reduceDemo() {
        System.out.println("Factorial (implicit identity):");
        // single param -> first element plays a role of identity (it is implicit), 
        // accumulator is (a, b) -> a * b -> returns optional
        IntStream.rangeClosed(1, 5)
                .reduce((a, b) -> a * b)
                .ifPresent(System.out::println);

        // with first param as identity
        // accumulator is (1, (a, b) -> a * b) -> returns value
        System.out.println("Factorial (explicit identity):");
        int reduced = IntStream.rangeClosed(1, 5)
                .reduce(1, (a, b) -> a * b);
        System.out.println(reduced);

        // three params (with combiner) - first param as identity (0, ...)
        // accumulator is (a, b) -> a + b)
        // combiner is Integer::sum
        // we must have identifier, and we can use parallel stream!
        Integer sum = Stream.iterate(1, n -> ++n)
                .limit(100000)
                .parallel().reduce(0, (a, b) -> a + b, Integer::sum);
        System.out.println("Sum: " + sum);
    }

    // Collect is an extremely useful terminal operation to transform the elements of the stream 
    // into a different kind of result, e.g. a List, Set or Map.
    // Collect accepts a Collector which consists of four different operations: 
    //  - supplier 
    //  - accumulator
    //  - combiner
    //  - finisher
    private static void collectDemo() {
        List<Person> people = Arrays.asList(
                new Person("Milica", 18),
                new Person("Danica", 23),
                new Person("Gojko", 24),
                new Person("Gojko", 33),
                new Person("Milutin", 12));

        Set<Person> adults = people.stream()
                .filter(p -> p.age >= 18)
                .collect(Collectors.toSet());
        System.out.println("Adults: " + adults);

        List<Person> sorted = people.stream()
                //.sorted()
                //.sorted(Comparator.comparing(Person::getName).thenComparing(Person::getAge))
                //.sorted(Comparator.comparing(Person::getName).reversed().thenComparing(Person::getAge))
                .sorted(Comparator.comparing(Person::getName).thenComparing(Person::getAge).reversed())
                .collect(Collectors.toList());
        System.out.println("Sorted: " + sorted);

        Map<Integer, String> ageNameMap = people.stream()
                .collect(Collectors.toMap(Person::getAge, Person::getName));
        System.out.println("Age name map: " + ageNameMap);

        // duplicate keys... it breaks!!
        /*
        Map<String, Integer> nameAgeMap = people.stream()
        .collect(Collectors.toMap(Person::getName, Person::getAge));
        System.out.println("Age name map: " + ageNameMap);
         */
        Map<String, List<Person>> namesMap = people.stream()
                .collect(Collectors.groupingBy(Person::getName));
        System.out.println("Names map:" + namesMap);

        String joined = people.stream()
                .map(Person::toString)
                .collect(Collectors.joining(", "));
        System.out.println("Joined persons: " + joined);

        Double averageAge = people.stream()
                .collect(Collectors.averagingInt(Person::getAge));
        System.out.println("Average age: " + averageAge);

        IntSummaryStatistics statistics = people.stream()
                .collect(Collectors.summarizingInt(Person::getAge));
        System.out.println("Statistics: " + statistics);
    }

    private static void parallelDemo() {
        Instant start = Instant.now();
        long sum = LongStream.range(1, 1_000_000_000).reduce(0, Long::sum); // 1_000 - better result
        Instant end = Instant.now();
        System.out.printf("Serial sum: %d, Time elapsed: %d ms%n", sum, Duration.between(start, end).toMillis());

        start = Instant.now();
        sum = LongStream.range(1, 1_000_000_000).parallel().reduce(0, Long::sum);  // 1_000 - worse result
        end = Instant.now();
        System.out.printf("Parallel sum: %d, Time elapsed: %d ms%n", sum, Duration.between(start, end).toMillis());
    }

    private static class Person implements Comparable<Person> {

        private final String name;
        private final int age;

        public Person(String name, int age) {
            this.name = name;
            this.age = age;
        }

        @Override
        public String toString() {
            return "Person{" + "name=" + name + ", age=" + age + '}';
        }

        public String getName() {
            return name;
        }

        public int getAge() {
            return age;
        }

        @Override
        public int compareTo(Person o) {
            return -getName().compareTo(o.name);
        }

        @Override
        public int hashCode() {
            int hash = 5;
            hash = 47 * hash + Objects.hashCode(this.name);
            return hash;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            final Person other = (Person) obj;
            return Objects.equals(this.name, other.name);
        }

    }
}
