package hr.algebra.generics.variance;

import java.util.Arrays;
import java.util.List;

public class InvariantNonReifiableGenerics {
    public static void main(String[] args) {
        List<Number> numbers = List.of(1, Math.PI, (byte)1); // returns immutable collection -> not elements, not size can change

        // generic types are NON-REIFIABLE, since at run time we cannot determine the true nature of the generic type
        // TYPE ERASURE - type information for type parameters is discarded by the compiler after the compilation of code is done
        List<Integer> integers = List.of(1, 2, 3);

        // Generics are INVARIANT:
        // List<Integer> does not extend List<Number>
        //List<Number> numbers = integers; //compiler error
//        numbers.add(Math.PI); // this would be a HEAP POLLUTION
        System.out.println("Sum: " + sum(integers));
        System.out.println("Sum: " + sum(Arrays.asList(2.3, 3.6)));
        // this does not pass!
        //System.out.println("Sum: " + sum(Arrays.asList("a", "a")));

        // but, it is possible to produce number from the list of integers
        Number n = getLast(integers);
        System.out.println(n);

        // this cannot pass, because of invariance - we have prevented it
        //setLast(numbers, 2.4);
        //setLast(integers, 2.4);

        // but who can consume?
        List<Object> objects = Arrays.asList(1, 2, 3); // returns ArrayList as [] that can change the elements but not it's size
        setLast(objects, 2.4);
    }

    // this does not pass, as it does with arrays
    // private static int sum(List<Number> numbers) {
    // Producer extends, because it can read the doubleValue() of Number and all his subtypes!
    // PRODUCER READS and therefore MUST EXTEND
    // this is enabling COVARIANCE
    private static double sum(List<? extends Number> numbers) {
        double sum = 0;
        for (Number number : numbers) {
            //here we are reading values! producer in action!
            sum += number.doubleValue();
        }
        // in producer, no changes are possible, it is read only!
        //numbers.add(1);
        return sum;
    }


    // this is enabling COVARIANCE
    private static Number getLast(List<? extends Number> numbers) {
        // here we are writing (producing) values! producer in action!
        return numbers.getLast();
    }

    // Consumer would like to take Number values, so it must be a Number or its supertype
    // otherwise we could pass this method a List<Integer> and values of Number -> Number can and do not have to be Integer
    // CONSUMER WRITES and therefore must SUPER
    // this is enabling CONTRA-VARIANCE
    private static void setLast(List<? super Number> numbers, Number number) {
        // here we are writing (consuming) values! consumer in action!
        numbers.set(numbers.size() - 1, number);
    }

}
