package hr.algebra.parsers;

import hr.algebra.model.Employee;
import hr.algebra.model.EmployeeTag;
import hr.algebra.model.EmployeeType;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import java.io.File;
import java.io.IOException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

public class SAXParserDemo {
    private static final String FILENAME = "/employees.xml";

    public static void main(String[] args) {
        try {
            // fail fast
//            String path = Objects.requireNonNull(SAXParserDemo.class.getResource(FILENAME)).getFile();
            String path = URLDecoder.decode(
                Objects.requireNonNull(SAXParserDemo.class.getResource(FILENAME)).getFile(),
                StandardCharsets.UTF_8
            );
            List<Employee> employees = parse(path);
            System.out.println(employees);
        } catch (ParserConfigurationException | SAXException | IOException e) {
            e.printStackTrace();
        }
    }

    private static List<Employee> parse(String path) throws ParserConfigurationException, SAXException, IOException {
        List<Employee> employees = new ArrayList<>();
        SAXParserFactory factory = SAXParserFactory.newInstance();
        factory.setValidating(true);  // comment out and do a mistake in XML
        SAXParser parser = factory.newSAXParser();
        parser.parse(new File(path), new EmployeesHandler(employees)); // pass by reference
        return employees;
    }

    private static class EmployeesHandler extends DefaultHandler {

        private final List<Employee> employees;

        private Optional<EmployeeTag> tag;
        private Employee employee;

        public EmployeesHandler(List<Employee> employees) {
            this.employees = employees; // this should be defensive copy, but we use it internally, and for the means of pass by value so ok!
        }

        @Override
        public void startDocument() throws SAXException {
            tag = Optional.empty();
        }

        @Override
        public void endDocument() throws SAXException {
            System.out.println("Done");
        }

        // qName is the full, qualified name, including both the namespace prefix, if any, and the localName.
        // So, for abc:xyz
        //	 The namespace prefix is abc.
        //	 The localName is xyz.
        //	 The qName is abc:xyz.
        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
            tag = EmployeeTag.of(qName);
            if (tag.isPresent() && tag.get().equals(EmployeeTag.EMPLOYEE)) {
                employee = new Employee();
                // actually Enum hast this already but is exception driven!
                employee.setEmployeeType(Enum.valueOf(EmployeeType.class, attributes.getValue(0)));
                employees.add(employee);
            }
        }

        @Override
        public void endElement(String uri, String localName, String qName) throws SAXException {
            if (tag.isPresent() && tag.get().equals(EmployeeTag.EMPLOYEE)) {
                employee = null;
            }
            tag = Optional.empty();
        }

        @Override
        public void characters(char[] ch, int start, int length) throws SAXException {
            String value = new String(ch, start, length);
            if (employee != null && tag.isPresent()) {
                switch (tag.get()) {
                    case ID -> employee.setId(Integer.parseInt(value));
                    case FIRSTNAME -> employee.setFirstName(value);
                    case LASTNAME -> employee.setLastName(value);
                    case INCOME -> employee.setIncome(Double.parseDouble(value));
                }
            }
        }

        @Override
        public void warning(SAXParseException e) throws SAXException {
            System.err.println("Warning: " + e.getMessage());
        }

        // we want to make sure everything works
        @Override
        public void error(SAXParseException e) throws SAXException {
            throw e;
        }
    }
}
