package hr.algebra.parsers;

import hr.algebra.model.Employee;
import hr.algebra.model.EmployeeTag;
import hr.algebra.model.EmployeeType;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Characters;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

public class StAXParserDemo {
    private static final String FILENAME = "/employees.xml";
    private static final String TYPE = "type";

    public static void main(String[] args) {
        try {
//            String path = Objects.requireNonNull(SAXParserDemo.class.getResource(FILENAME)).getFile();
            String path = URLDecoder.decode(
                    Objects.requireNonNull(SAXParserDemo.class.getResource(FILENAME)).getFile(),
                    StandardCharsets.UTF_8
            );
            List<Employee> employees = parse(path);
            System.out.println(employees);
        } catch (FileNotFoundException | XMLStreamException e) {
            e.printStackTrace();
        }
    }

    private static List<Employee> parse(String path) throws FileNotFoundException, XMLStreamException {
        List<Employee> employees = new ArrayList<>();
        XMLInputFactory factory = XMLInputFactory.newFactory();
        XMLEventReader reader = factory.createXMLEventReader(new FileInputStream(path));
        parse(reader, employees); // pass by reference
        return employees;
    }


    private static void parse(XMLEventReader reader, List<Employee> employees) throws XMLStreamException {
        Optional<EmployeeTag> tag = Optional.empty();
        Employee employee = null;
        while (reader.hasNext()) {
            XMLEvent event = reader.nextEvent();
            switch (event.getEventType()) {
                case XMLStreamConstants.START_ELEMENT -> {
                    StartElement startElement = event.asStartElement();
                    tag = EmployeeTag.of(startElement.getName().getLocalPart());
                    if (tag.isPresent() && tag.get().equals(EmployeeTag.EMPLOYEE)) {
                        employee = new Employee();
                        String employeeType = startElement.getAttributeByName(QName.valueOf(TYPE)).getValue();
                        employee.setEmployeeType(Enum.valueOf(EmployeeType.class, employeeType));
                        employees.add(employee);
                    }
                }
                case XMLStreamConstants.CHARACTERS -> {
                    if (employee != null && tag.isPresent()) {
                        Characters characters = event.asCharacters();
                        String value = characters.getData();
                        switch (tag.get()) {
                            case ID -> employee.setId(Integer.parseInt(value));
                            case FIRSTNAME -> employee.setFirstName(value);
                            case LASTNAME -> employee.setLastName(value);
                            case INCOME -> employee.setIncome(Double.parseDouble(value));
                        }
                    }
                }
                case XMLStreamConstants.END_ELEMENT -> tag = Optional.empty();
            }
        }
    }
}
