package hr.algebra.books.controller;

import hr.algebra.books.BooksApplication;
import javafx.application.Platform;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.collections.transformation.FilteredList;
import javafx.collections.transformation.SortedList;
import javafx.event.ActionEvent;
import javafx.fxml.FXMLLoader;
import javafx.fxml.Initializable;
import javafx.scene.Scene;
import javafx.scene.control.*;
import javafx.scene.control.cell.PropertyValueFactory;
import javafx.scene.control.cell.TextFieldTableCell;
import javafx.scene.control.skin.TableColumnHeader;
import javafx.scene.input.MouseEvent;
import javafx.stage.Stage;
import model.Book;
import repository.BooksRepository;

import javax.swing.table.TableColumnModel;
import java.io.IOException;
import java.net.URL;
import java.util.Optional;
import java.util.ResourceBundle;

public class BooksController implements Initializable {
    public TableView<Book> tvBooks;
    public TableColumn<Book, String> tcId, tcTitle, tcAuthor;
    public TextField tfFilter;

    private ObservableList<Book> books;


    public void exitApp() {
        Platform.exit();
    }

    public void showBookDetails(MouseEvent mouseEvent) {
        if (mouseEvent.getClickCount() != 2
                || mouseEvent.getTarget().getClass().equals(TableColumnHeader.class))  {
            return;
        }
        Book book = tvBooks.getSelectionModel().getSelectedItem();
        if (book != null) {
            try {
                redirectToBookDetails(book);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    private void redirectToBookDetails(Book book) throws IOException {
        FXMLLoader fxmlLoader = new FXMLLoader(BooksApplication.class.getResource("view/BookDetails.fxml"));
        Scene scene = new Scene(fxmlLoader.load());
        Stage stage = new Stage();
        stage.setTitle("Book details");
        stage.setScene(scene);

        BookDetailsController bookDetailsController = fxmlLoader.getController();
        bookDetailsController.initData(book);

        stage.show();
    }

    public void filterBooks() {
        showBooks();
    }

    @Override
    public void initialize(URL url, ResourceBundle resourceBundle) {
        initTableCells();
        books = FXCollections.observableArrayList(BooksRepository.getBooks());
        showBooks();
        if (!books.isEmpty()) {
            tvBooks.getSelectionModel().select(0);
        }
    }

    private void initTableCells() {
        tcId.setCellValueFactory(new PropertyValueFactory<>("id"));
        tcTitle.setCellValueFactory(new PropertyValueFactory<>("title"));
        tcTitle.setCellFactory(TextFieldTableCell.forTableColumn());
        tcTitle.setOnEditCommit(event -> {
            Book book = tvBooks.getSelectionModel().getSelectedItem();
            book.setTitle(event.getNewValue());
            showBooks();
        });

        tcAuthor.setCellValueFactory(new PropertyValueFactory<>("author"));
        tcAuthor.setCellFactory(TextFieldTableCell.forTableColumn());
        tcAuthor.setOnEditCommit(event -> tvBooks.getSelectionModel().getSelectedItem().setAuthor(event.getNewValue()));
    }

    private void showBooks() {
        FilteredList<Book> filteredBooks = books.filtered(b -> b.getTitle()
                .toLowerCase()
                .startsWith(tfFilter.getText().toLowerCase()));
        SortedList<Book> sortedBooks = new SortedList<>(filteredBooks);
        tvBooks.setItems(sortedBooks);
        sortedBooks.comparatorProperty().bind(tvBooks.comparatorProperty());
    }

    public void deleteBook(ActionEvent actionEvent) {
        Book book = tvBooks.getSelectionModel().getSelectedItem();
        if (book != null) {
            Alert alert = new Alert(Alert.AlertType.CONFIRMATION);
            alert.setTitle("Confirmation");
            alert.setHeaderText("Delete book?");
            alert.setContentText("Really tired of ''" + book + "''?");
            Optional<ButtonType> confirmed = alert.showAndWait();
            if (confirmed.isPresent() && confirmed.get() == ButtonType.OK) {
                books.remove(book);
            }
        }
    }
}
