﻿using AutoMapper;
using ex13.BL.Models;
using ex13.BL.Services;
using exercise_13.Services;
using exercise_13.ViewModels;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;

namespace exercise_13.Controllers
{
    public class SongController : Controller
    {
        private readonly Exercise13Context _context;
        private readonly IConfiguration _configuration;
        private readonly IMapper _mapper;
        private readonly IAudioRepository _audioRepo;
        private readonly ICaching _caching;

        public SongController(
            Exercise13Context context, 
            IConfiguration configuration, 
            IMapper mapper,
            IAudioRepository audioRepo,
            ICaching caching)
        {
            _context = context;
            _configuration = configuration;
            _mapper = mapper;
            _audioRepo = audioRepo;
            _caching = caching;
        }

        // GET: SongController
        public ActionResult Index()
        {
            try
            {
                var songs = _audioRepo.GetAll();
                var songVms = _mapper.Map<IEnumerable<SongVM>>(songs);

                return View(songVms);
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        public ActionResult GetSongsByDuration(int? min, int? max)
        {
            try
            {
                IEnumerable<Audio> songs = _context.Audios
                    .Include(x => x.Genre)
                    .Include(x => x.Artist);

                if (min.HasValue)
                {
                    songs = songs.Where(x => x.Duration >= min.Value);
                }

                if (max.HasValue)
                {
                    songs = songs.Where(x => x.Duration <= max.Value);
                }

                var songVms =
                    songs.Select(x => new SongVM
                    {
                        Id = x.Id,
                        Title = x.Title,
                        Year = x.Year,
                        ArtistId = x.ArtistId,
                        ArtistName = x.Artist.Name,
                        GenreId = x.GenreId,
                        GenreName = x.Genre.Name,
                        Duration = x.Duration,
                        Url = x.Url
                    })
                    .ToList();

                return View("Index", songVms);
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        public ActionResult Search(SearchVM searchVm)
        {
            try
            {
                if (string.IsNullOrEmpty(searchVm.Q) && string.IsNullOrEmpty(searchVm.Submit))
                {
                    searchVm.Q = Request.Cookies["query"];
                }

                IQueryable<Audio> songs = _context.Audios
                    .Include(x => x.Genre)
                    .Include(x => x.Artist);

                if (!string.IsNullOrEmpty(searchVm.Q))
                {
                    songs = songs.Where(x => x.Title.Contains(searchVm.Q));
                }

                // We need this for pager
                var filteredCount = songs.Count();

                switch (searchVm.OrderBy.ToLower())
                {
                    case "id":
                        songs = songs.OrderBy(x => x.Id);
                        break;
                    case "title":
                        songs = songs.OrderBy(x => x.Title);
                        break;
                    case "year":
                        songs = songs.OrderBy(x => x.Year);
                        break;
                    case "duration":
                        songs = songs.OrderBy(x => x.Duration);
                        break;
                    case "genre":
                        songs = songs.OrderBy(x => x.Genre.Name);
                        break;
                    case "artist":
                        songs = songs.OrderBy(x => x.Artist.Name);
                        break;
                }

                songs = songs.Skip((searchVm.Page - 1) * searchVm.Size).Take(searchVm.Size); // if pages start from 1

                searchVm.Songs =
                    songs.Select(x => new SongVM
                    {
                        Id = x.Id,
                        Title = x.Title,
                        Year = x.Year,
                        ArtistId = x.ArtistId,
                        ArtistName = x.Artist.Name,
                        GenreId = x.GenreId,
                        GenreName = x.Genre.Name,
                        Duration = x.Duration,
                        Url = x.Url
                    })
                    .ToList();

                // BEGIN PAGER
                var expandPages = _configuration.GetValue<int>("Paging:ExpandPages");
                searchVm.LastPage = (int)Math.Ceiling(1.0 * filteredCount / searchVm.Size);
                searchVm.FromPager = searchVm.Page > expandPages ? 
                    searchVm.Page - expandPages : 
                    1;
                searchVm.ToPager = (searchVm.Page + expandPages) < searchVm.LastPage ? 
                    searchVm.Page + expandPages : 
                    searchVm.LastPage;
                // END PAGER

                var option = new CookieOptions { Expires = DateTime.Now.AddMinutes(15) };
                Response.Cookies.Append("query", searchVm.Q ?? "", option);
                
                return View(searchVm);
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        // GET: SongController/Details/5
        public ActionResult Details(int id)
        {
            return View();
        }

        // GET: SongController/Create
        public ActionResult Create()
        {
            try
            {
                ViewBag.GenreDdlItems = _caching.GetGenreListItems();
                ViewBag.ArtistDdlItems = _caching.GetArtistListItems();

                var song = new SongVM();
                int.TryParse(Request.Cookies["SongYear"], out int year);
                song.Year = year == 0 ? null : year;

                return View(song);
            }
            catch (Exception)
            {

                throw;
            }
        }

        // POST: SongController/Create
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult Create(SongVM song)
        {
            try
            {
                if (!ModelState.IsValid) 
                {
                    ViewBag.GenreDdlItems = _caching.GetGenreListItems();
                    ViewBag.ArtistDdlItems = _caching.GetArtistListItems();

                    ModelState.AddModelError("", "Failed to create song");

                    return View();
                }

                _audioRepo.Add(
                    song.Title,
                    song.Year,
                    song.GenreId,
                    song.ArtistId,
                    song.Duration,
                    song.Url);

                var option = new CookieOptions { Expires = DateTime.Now.AddDays(14) };
                Response.Cookies.Append("SongYear", song.Year.ToString(), option);

                return RedirectToAction(nameof(Index));
            }
            catch
            {
                return View();
            }
        }

        // GET: SongController/Edit/5
        public ActionResult Edit(int id)
        {
            ViewBag.GenreDdlItems = _caching.GetGenreListItems();
            ViewBag.ArtistDdlItems = _caching.GetArtistListItems();
            ViewBag.TagDdlItems = _caching.GetTagListItems();

            var song = _audioRepo.Get(id);
            var songVM = _mapper.Map<SongVM>(song);

            return View(songVM);
        }

        // POST: SongController/Edit/5
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult Edit(int id, SongVM song)
        {
            try
            {
                _audioRepo.Modify(
                    id,
                    song.Title,
                    song.Year,
                    song.GenreId,
                    song.ArtistId,
                    song.Duration,
                    song.Url,
                    song.TagIds);

                return RedirectToAction(nameof(Index));
            }
            catch(Exception ex)
            {
                return View();
            }
        }

        // GET: SongController/Delete/5
        public ActionResult Delete(int id)
        {
            var song = _audioRepo.Get(id);
            var songVM = _mapper.Map<SongVM>(song);

            return View(songVM);
        }

        // POST: SongController/Delete/5
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult Delete(int id, Audio audio)
        {
            try
            {
                _audioRepo.Remove(id);

                return RedirectToAction(nameof(Index));
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }
    }
}
