﻿using exercise_3_4.Models;
using exercise_3_4.Repository;
using Microsoft.AspNetCore.Mvc;

// For more information on enabling Web API for empty projects, visit https://go.microsoft.com/fwlink/?LinkID=397860

namespace exercise_3_4.Controllers
{
    [Route("api/[controller]")]
    [ApiController]
    public class AddressController : ControllerBase
    {
        // GET: api/<AddressController>
        [HttpGet]
        public ActionResult<IEnumerable<Street>> Get()
        {
            try
            {
                return Ok(AddressRepository.Streets);
            }
            catch (Exception)
            {
                // TODO: Log error, use details from the exception

                return StatusCode(500, "There was a problem while retrieving streets");
            }
        }

        // GET api/<AddressController>/5
        [HttpGet("{id}")]
        public ActionResult<HouseNumber> Get(int id)
        {
            try
            {
                var street = AddressRepository.Streets.FirstOrDefault(x => x.Id == id);
                if (street == null)
                {
                    return NotFound();
                }

                return Ok(street);
            }
            catch (Exception)
            {
                // TODO: Log error, use details from the exception

                return BadRequest("There was a problem while updating street");
            }
        }

        [HttpGet("HouseNumbers/{id}")]
        public ActionResult<IEnumerable<HouseNumber>> GetHouseNumbers(int id)
        {
            try
            {
                var street = AddressRepository.Streets.FirstOrDefault(x => x.Id == id);
                if (street == null)
                {
                    return NotFound();
                }

                var result = street.HouseNumbers.OrderBy(x => x.Number).ThenBy(x => x.Addendum);

                return Ok(result);
            }
            catch (Exception)
            {
                // TODO: Log error, use details from the exception

                return BadRequest("There was a problem while updating street");
            }
        }

        [HttpGet("[action]")]
        public ActionResult<IEnumerable<Street>> Search(string text, int sortType = 0, int start = 0, int count = 10)
        {
            try
            {
                IEnumerable<Street> results = AddressRepository.Streets;

                // Filtering
                if (!String.IsNullOrEmpty(text))
                {
                    results = results.Where(s => s.Name.Contains(text, StringComparison.OrdinalIgnoreCase));
                }

                // Sorting
                // - sortType = 0: Id
                // - sortType = 1: Name
                if (sortType == 0)
                {
                    results = results.OrderBy(s => s.Id);
                }
                else if (sortType == 1)
                {
                    results = results.OrderBy(s => s.Name);
                }

                // Paging
                results = results.Skip(start * count).Take(count);

                return Ok(results);
            }
            catch (Exception)
            {
                return StatusCode(500);
            }
        }

        // POST api/<AddressController>
        [HttpPost]
        public ActionResult<Street> Post([FromBody] Street street)
        {
            try
            {
                if(String.IsNullOrEmpty(street.Name))
                    return BadRequest("Name cannot be empty");

                int maxId = AddressRepository.Streets.Any() ? AddressRepository.Streets.Max(x => x.Id) : 0;
                street.Id = maxId + 1;

                AddressRepository.Streets.Add(street);

                var location = Url.Action(nameof(Get), new { id = street.Id });
                return Created(location, street);
            }
            catch (Exception)
            {
                // TODO: Log error, use details from the exception

                return BadRequest("There was a problem while updating street");
            }
        }

        // PUT api/<AddressController>/5
        [HttpPut("{id}")]
        public ActionResult<Street> Put(int id, [FromBody] Street street)
        {
            try
            {
                var existingStreet = AddressRepository.Streets.FirstOrDefault(x => x.Id == id);
                if (existingStreet == null)
                {
                    return NotFound();
                }

                existingStreet.Name = street.Name;

                return Ok(existingStreet);
            }
            catch (Exception)
            {
                // TODO: Log error, use details from the exception

                return BadRequest("There was a problem while updating street");
            }
        }

        // DELETE api/<AddressController>/5
        [HttpDelete("{id}")]
        public ActionResult<Street> Delete(int id)
        {
            try
            {
                var existingStreet = AddressRepository.Streets.FirstOrDefault(x => x.Id == id);
                if (existingStreet == null)
                {
                    return NotFound();
                }

                AddressRepository.Streets.Remove(existingStreet);

                return Ok(existingStreet);
            }
            catch (Exception)
            {
                // TODO: Log error, use details from the exception

                return BadRequest("There was a problem while updating street");
            }
        }
    }
}
