﻿using exercise_6_7.Dtos;
using exercise_6_7.Models;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;

// For more information on enabling Web API for empty projects, visit https://go.microsoft.com/fwlink/?LinkID=397860

namespace exercise_6_7.Controllers
{
    [Route("api/[controller]")]
    [ApiController]
    public class ProductController : ControllerBase
    {
        private readonly IConfiguration _configuration;
        private readonly Exercise6Context _context;

        public ProductController(IConfiguration configuration, Exercise6Context context)
        {
            _configuration = configuration;
            _context = context;
        }

        [Authorize(Roles = "Admin,User")]
        [HttpGet]
        public ActionResult<IEnumerable<ProductDto>> Get()
        {
            try
            {
                var result = MapToDto(_context.Products);

                return Ok(result);
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        [Authorize(Roles = "Admin,User")]
        [HttpGet("{id}")]
        public ActionResult<ProductDto> Get(int id)
        {
            try
            {
                var product = _context.Products.FirstOrDefault(x => x.Id == id);
                if(product == null)
                    return NotFound();

                var result = MapToDto(product);

                return Ok(result);
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        // POST api/<ProductController>
        [Authorize(Roles = "Admin")]
        [HttpPost]
        public ActionResult<ProductDto> Post([FromBody] ProductDto value)
        {
            try
            {
                var product = MapFromDto(value);
                _context.Products.Add(product);
                _context.SaveChanges();
                
                var result = MapToDto(product);
                return Ok(result);
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        // PUT api/<ProductController>/5
        [Authorize(Roles = "Admin")]
        [HttpPut("{id}")]
        public ActionResult<ProductDto> Put(int id, [FromBody] ProductDto value)
        {
            try
            {
                var product = _context.Products.FirstOrDefault(x => x.Id == id);
                if (product == null)
                    return NotFound();

                product.Title = value.Title;
                product.Price = value.Price;

                _context.SaveChanges();

                var result = MapToDto(product);
                return Ok(result);
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        // DELETE api/<ProductController>/5
        [Authorize(Roles = "Admin")]
        [HttpDelete("{id}")]
        public ActionResult<ProductDto> Delete(int id)
        {
            try
            {
                var product = _context.Products.FirstOrDefault(x => x.Id == id);
                if (product == null)
                    return NotFound();

                _context.Products.Remove(product);
                _context.SaveChanges();

                var result = MapToDto(product);
                return Ok(result);
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        private IEnumerable<ProductDto> MapToDto(IEnumerable<Product> products) =>
            products.Select(x => new ProductDto
            {
                Id = x.Id,
                Title = x.Title,
                Price = x.Price,
            });

        private ProductDto MapToDto(Product product) =>
            new ProductDto
            {
                Id = product.Id,
                Title = product.Title,
                Price = product.Price
            };

        private Product MapFromDto(ProductDto product) =>
            new Product
            {
                Id = product.Id,
                Title = product.Title,
                Price = product.Price
            };
    }
}
