﻿using exercise_6_7.Dtos;
using exercise_6_7.Models;
using exercise_6_7.Security;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;

namespace exercise_6_7.Controllers
{
    [Route("api/[controller]")]
    [ApiController]
    public class UserController : ControllerBase
    {
        private readonly IConfiguration _configuration;
        private readonly Exercise6Context _context;

        public UserController(IConfiguration configuration, Exercise6Context context)
        {
            _configuration = configuration;
            _context = context;
        }

        [HttpPost("[action]")]
        public ActionResult<UserRegisterDto> Register(UserRegisterDto registerDto)
        {
            try
            {
                // Check if there is such a username in the database already
                var trimmedUsername = registerDto.Username.Trim();
                if (_context.Users.Any(x => x.Username.Equals(trimmedUsername)))
                    return BadRequest($"Username {trimmedUsername} already exists");

                var userRole = _context.Roles.FirstOrDefault(x => x.Name == "User");

                // Hash the password
                var b64salt = PasswordHashProvider.GetSalt();
                var b64hash = PasswordHashProvider.GetHash(registerDto.Password, b64salt);

                // Create user from DTO and hashed password
                var user = new User
                {
                    Id = registerDto.Id,
                    Username = registerDto.Username,
                    PwdHash = b64hash,
                    PwdSalt = b64salt,
                    FirstName = registerDto.FirstName,
                    LastName = registerDto.LastName,
                    Email = registerDto.Email,
                    Phone = registerDto.Phone,
                    RoleId = userRole.Id,
                };

                // Add user and save changes to database
                _context.Add(user);
                _context.SaveChanges();

                // Update DTO Id to return it to the client
                registerDto.Id = user.Id;

                return Ok(registerDto);

            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        [HttpPost("[action]")]
        public ActionResult Login(UserSignInDto signInDto)
        {
            try
            {
                var genericLoginFail = "Incorrect username or password";

                // Try to get a user from database
                var existingUser = _context.Users.Include(x => x.Role).FirstOrDefault(x => x.Username == signInDto.Username);
                if (existingUser == null)
                    return BadRequest(genericLoginFail);

                // Check is password hash matches
                var b64hash = PasswordHashProvider.GetHash(signInDto.Password, existingUser.PwdSalt);
                if (b64hash != existingUser.PwdHash)
                    return BadRequest(genericLoginFail);

                // Create and return JWT token
                var secureKey = _configuration["JWT:SecureKey"];

                /* This is a hardcoded role example!
                var serializedToken =
                    JwtTokenProvider.CreateToken(
                        secureKey,
                        120,
                        signInDto.Username,
                        existingUser.Username == "Admin" ? "Admin" : "User");
                */

                var serializedToken =
                    JwtTokenProvider.CreateToken(
                        secureKey,
                        120,
                        signInDto.Username,
                        existingUser.Role.Name);

                return Ok(serializedToken);
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        [HttpPost("[action]")]
        public ActionResult<UserChangePasswordDto> ChangePassword(UserChangePasswordDto changePasswordDto)
        {
            try
            {
                // Check if there is such a username in the database already
                var trimmedUsername = changePasswordDto.Username.Trim();
                var existingUser = _context.Users.FirstOrDefault(x => x.Username.Equals(trimmedUsername));
                if (existingUser == null)
                    return BadRequest($"Username {trimmedUsername} does not exist");

                // Hash the password
                existingUser.PwdSalt = PasswordHashProvider.GetSalt();
                existingUser.PwdHash = PasswordHashProvider.GetHash(changePasswordDto.Password, existingUser.PwdSalt);

                // Save changes to database
                _context.SaveChanges();

                return Ok();

            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        [HttpPost("[action]")]
        public ActionResult<UserRegisterDto> PromoteUser(UserPromoteDto promoteDto)
        {
            try
            {
                // Check if there is such a username in the database already
                var trimmedUsername = promoteDto.Username.Trim();
                var existingUser = _context.Users.FirstOrDefault(x => x.Username.Equals(trimmedUsername));
                if (existingUser == null)
                    return BadRequest($"Username {trimmedUsername} does not exist");

                var adminRole = _context.Roles.FirstOrDefault(x => x.Name == "Admin");

                // Update role
                existingUser.RoleId = adminRole.Id;

                // Save changes to database
                _context.SaveChanges();

                return Ok();
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }


    }
}
