﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;
using Task07.Mapping;
using Task07.Models;

namespace Task07.Controllers
{
    [Route("api/[controller]")]
    [ApiController]
    public class AudioController : ControllerBase
    {
        private readonly Task07Context _dbContext;

        public AudioController(Task07Context dbContext)
        {
            _dbContext = dbContext;
        }

        [HttpGet("[action]")]
        public ActionResult<IEnumerable<BLAudio>> GetAll()
        {
            try
            {
                var dbAudios = _dbContext.Audios.Include("Genre").Include("AudioTags").Include("AudioTags.Tag");

                var audios = AudioMapping.MapToBL(dbAudios);

                return Ok(audios);
            }
            catch (Exception ex)
            {
                return StatusCode(
                    StatusCodes.Status500InternalServerError, 
                    "There has been a problem while fetching the data you requested");
            }
        }

        [HttpGet("[action]")]
        public ActionResult<IEnumerable<BLAudio>> Search(string searchPart)
        {
            try
            {
                var dbAudios = _dbContext.Audios.Where(x => x.Title.Contains(searchPart));

                var audios = AudioMapping.MapToBL(dbAudios);

                return Ok(audios);
            }
            catch (Exception ex)
            {
                return StatusCode(
                    StatusCodes.Status500InternalServerError,
                    "There has been a problem while fetching the data you requested");
            }
        }

        [HttpGet("{id}")]
        public ActionResult<BLAudio> Get(int id)
        {
            try
            {
                var dbAudio = _dbContext.Audios.FirstOrDefault(x => x.Id == id);
                if (dbAudio == null)
                    return NotFound($"Could not find audio with id {id}");

                var audio = AudioMapping.MapToBL(dbAudio);

                return Ok(audio);
            }
            catch (Exception ex)
            {
                return StatusCode(
                    StatusCodes.Status500InternalServerError,
                    "There has been a problem while fetching the data you requested");
            }
        }

        [HttpPost()]
        public ActionResult<BLAudio> Post(BLAudio audio)
        {
            try
            {
                if (!ModelState.IsValid)
                    return BadRequest(ModelState);

                var dbAudio = AudioMapping.MapToDAL(audio);

                var dbTags = _dbContext.Tags.Where(x => audio.Tags.Contains(x.Name));
                dbAudio.AudioTags = dbTags.Select(x => new AudioTag { Tag = x }).ToList();

                _dbContext.Audios.Add(dbAudio);

                _dbContext.SaveChanges();

                audio = AudioMapping.MapToBL(dbAudio);

                return Ok(audio);
            }
            catch (Exception ex)
            {
                return StatusCode(
                    StatusCodes.Status500InternalServerError,
                    "There has been a problem while fetching the data you requested");
            }
        }

        [HttpPut("{id}")]
        public ActionResult<BLAudio> Put(int id, BLAudio audio)
        {
            try
            {
                if (!ModelState.IsValid)
                    return BadRequest(ModelState);

                var dbAudio = _dbContext.Audios.FirstOrDefault(x => x.Id == id);
                if (dbAudio == null)
                    return NotFound($"Could not find audio with id {id}");

                dbAudio.Title = audio.Title;
                dbAudio.GenreId = audio.GenreId;
                dbAudio.Duration = audio.Duration;
                dbAudio.Url = audio.Url;

                // (1) Remove unused tags
                var toRemove = dbAudio.AudioTags.Where(x => !audio.Tags.Contains(x.Tag.Name));
                foreach (var srTag in toRemove)
                {
                    _dbContext.AudioTags.Remove(srTag);
                }

                // (2) Add new tags
                var existingDbTagNames = dbAudio.AudioTags.Select(x => x.Tag.Name);
                var newTagNames = audio.Tags.Except(existingDbTagNames);
                foreach (var newTagName in newTagNames)
                {
                    var dbTag = _dbContext.Tags.FirstOrDefault(x => newTagName == x.Name);
                    // What if the tag doesn't exist at all?
                    if (dbTag == null)
                        continue;

                    dbAudio.AudioTags.Add(new AudioTag
                    {
                        Audio = dbAudio,
                        Tag = dbTag
                    });
                }

                _dbContext.SaveChanges();

                audio = AudioMapping.MapToBL(dbAudio);

                return Ok(audio);
            }
            catch (Exception ex)
            {
                return StatusCode(
                    StatusCodes.Status500InternalServerError,
                    "There has been a problem while fetching the data you requested");
            }
        }

        [HttpDelete("{id}")]
        public ActionResult<BLAudio> Delete(int id)
        {
            try
            {
                var dbAudio = _dbContext.Audios.FirstOrDefault(x => x.Id == id);
                if (dbAudio == null)
                    return NotFound($"Could not find audio with id {id}");

                _dbContext.Audios.Remove(dbAudio);

                var audio = AudioMapping.MapToBL(dbAudio);

                _dbContext.SaveChanges();

                return Ok(audio);
            }
            catch (Exception ex)
            {
                return StatusCode(
                    StatusCodes.Status500InternalServerError,
                    "There has been a problem while fetching the data you requested");
            }
        }

    }
}
