﻿using exercise_14.Models;
using exercise_14.ViewModels;
using Humanizer.Localisation;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Rendering;
using Microsoft.EntityFrameworkCore;
using Microsoft.EntityFrameworkCore.Metadata.Internal;
using Newtonsoft.Json.Linq;
using NuGet.Packaging;
using NuGet.Protocol;
using System.Diagnostics;
using System.Security.Cryptography;

namespace exercise_14.Controllers
{
    public class SongController : Controller
    {
        private readonly Exercise14Context _context;
        private readonly IConfiguration _configuration;

        public SongController(Exercise14Context context, IConfiguration configuration)
        {
            _context = context;
            _configuration = configuration;
        }

        private List<SelectListItem> GetGenreListItems()
        {
            var genreListItemsJson = HttpContext.Session.GetString("GenreListItems");

            List<SelectListItem> genreListItems;
            if (genreListItemsJson == null)
            {
                genreListItems = _context.Genres
                    .Select(x => new SelectListItem
                    {
                        Text = x.Name,
                        Value = x.Id.ToString()
                    }).ToList();

                HttpContext.Session.SetString("GenreListItems", genreListItems.ToJson());
            }
            else
            {
                genreListItems = genreListItemsJson.FromJson<List<SelectListItem>>();
            }

            return genreListItems;
        }

        private List<SelectListItem> GetArtistListItems()
        {
            var artistListItemsJson = HttpContext.Session.GetString("ArtistListItems");

            List<SelectListItem> artistListItems;
            if (artistListItemsJson == null)
            {
                artistListItems = _context.Artists
                    .Select(x => new SelectListItem
                    {
                        Text = x.Name,
                        Value = x.Id.ToString()
                    }).ToList();

                HttpContext.Session.SetString("ArtistListItems", artistListItems.ToJson());
            }
            else
            {
                artistListItems = artistListItemsJson.FromJson<List<SelectListItem>>();
            }

            return artistListItems;
        }

        private List<SelectListItem> GetTagListItems()
        {
            return _context.Tags
                .Select(x => new SelectListItem
                {
                    Text = x.Name,
                    Value = x.Id.ToString()
                }).ToList();
        }

        // GET: SongController
        public ActionResult Index()
        {
            try
            {
                var songVms = _context.Audios
                    .Include(x => x.Genre)
                    .Include(x => x.Artist)
                    .Select(x => new SongVM
                        {
                            Id = x.Id,
                            Title = x.Title,
                            Year = x.Year,
                            ArtistId = x.ArtistId,
                            ArtistName = x.Artist.Name,
                            GenreId = x.GenreId,
                            GenreName = x.Genre.Name,
                            Duration = x.Duration,
                            Url = x.Url
                    })
                    .ToList();

                return View(songVms);
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        public ActionResult GetSongsByDuration(int? min, int? max)
        {
            try
            {
                IEnumerable<Audio> songs = _context.Audios
                    .Include(x => x.Genre)
                    .Include(x => x.Artist);

                if (min.HasValue)
                {
                    songs = songs.Where(x => x.Duration >= min.Value);
                }

                if (max.HasValue)
                {
                    songs = songs.Where(x => x.Duration <= max.Value);
                }

                var songVms =
                    songs.Select(x => new SongVM
                    {
                        Id = x.Id,
                        Title = x.Title,
                        Year = x.Year,
                        ArtistId = x.ArtistId,
                        ArtistName = x.Artist.Name,
                        GenreId = x.GenreId,
                        GenreName = x.Genre.Name,
                        Duration = x.Duration,
                        Url = x.Url
                    })
                    .ToList();

                return View("Index", songVms);
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        public ActionResult Search(SearchVM searchVm)
        {
            try
            {
                PrepareSearchViewmodel(searchVm);

                return View(searchVm);
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        public ActionResult SearchPartial(SearchVM searchVm)
        {
            try
            {
                PrepareSearchViewmodel(searchVm);

                return PartialView("_SearchPartial", searchVm);
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private void PrepareSearchViewmodel(SearchVM searchVm)
        {
            IQueryable<Audio> songs = _context.Audios
                .Include(x => x.Genre)
                .Include(x => x.Artist);

            if (!string.IsNullOrEmpty(searchVm.Q))
            {
                songs = songs.Where(x => x.Title.Contains(searchVm.Q));
            }

            // We need this for pager
            var filteredCount = songs.Count();

            switch (searchVm.OrderBy.ToLower())
            {
                case "id":
                    songs = songs.OrderBy(x => x.Id);
                    break;
                case "title":
                    songs = songs.OrderBy(x => x.Title);
                    break;
                case "year":
                    songs = songs.OrderBy(x => x.Year);
                    break;
                case "duration":
                    songs = songs.OrderBy(x => x.Duration);
                    break;
                case "genre":
                    songs = songs.OrderBy(x => x.Genre.Name);
                    break;
                case "artist":
                    songs = songs.OrderBy(x => x.Artist.Name);
                    break;
            }

            songs = songs.Skip((searchVm.Page - 1) * searchVm.Size).Take(searchVm.Size); // if pages start from 1

            searchVm.Songs =
                songs.Select(x => new SongVM
                {
                    Id = x.Id,
                    Title = x.Title,
                    Year = x.Year,
                    ArtistId = x.ArtistId,
                    ArtistName = x.Artist.Name,
                    GenreId = x.GenreId,
                    GenreName = x.Genre.Name,
                    Duration = x.Duration,
                    Url = x.Url
                })
                .ToList();

            // BEGIN PAGER
            var expandPages = _configuration.GetValue<int>("Paging:ExpandPages");
            searchVm.LastPage = (int)Math.Ceiling(1.0 * filteredCount / searchVm.Size);
            searchVm.FromPager = searchVm.Page > expandPages ?
                searchVm.Page - expandPages :
                1;
            searchVm.ToPager = (searchVm.Page + expandPages) < searchVm.LastPage ?
                searchVm.Page + expandPages :
                searchVm.LastPage;
            // END PAGER
        }

        // GET: SongController/Details/5
        public ActionResult Details(int id)
        {
            return View();
        }

        // GET: SongController/Create
        public ActionResult Create()
        {
            try
            {
                ViewBag.GenreDdlItems = GetGenreListItems();
                ViewBag.ArtistDdlItems = GetArtistListItems();

                var song = new SongVM();
                int.TryParse(Request.Cookies["SongYear"], out int year);
                song.Year = year == 0 ? null : year;

                return View(song);
            }
            catch (Exception)
            {

                throw;
            }
        }

        // POST: SongController/Create
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult Create(SongVM song)
        {
            try
            {
                if (!ModelState.IsValid) 
                {
                    ViewBag.GenreDdlItems = GetGenreListItems();
                    ViewBag.ArtistDdlItems = GetArtistListItems();

                    ModelState.AddModelError("", "Failed to create song");

                    return View();
                }

                var newSong = new Audio
                {
                    Title = song.Title,
                    Year = song.Year,
                    GenreId = song.GenreId,
                    ArtistId = song.ArtistId,
                    Duration = song.Duration,
                    Url = song.Url
                };

                _context.Audios.Add(newSong);

                _context.SaveChanges();

                var option = new CookieOptions { Expires = DateTime.Now.AddDays(14) };
                Response.Cookies.Append("SongYear", song.Year.ToString(), option);

                return RedirectToAction(nameof(Index));
            }
            catch
            {
                return View();
            }
        }

        // GET: SongController/Edit/5
        public ActionResult Edit(int id)
        {
            ViewBag.GenreDdlItems = GetGenreListItems();
            ViewBag.ArtistDdlItems = GetArtistListItems();
            ViewBag.TagDdlItems = GetTagListItems();

            var song = _context.Audios.Include(x => x.AudioTags).FirstOrDefault(x => x.Id == id);
            var songVM = new SongVM
            {
                Id = song.Id,
                Title = song.Title,
                Year = song.Year,
                GenreId = song.GenreId,
                ArtistId = song.ArtistId,
                Duration = song.Duration,
                Url = song.Url,
                TagIds = song.AudioTags.Select(x => x.TagId).ToList()
            };

            return View(songVM);
        }

        // POST: SongController/Edit/5
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult Edit(int id, SongVM song)
        {
            try
            {
                var dbSong = _context.Audios.Include(x => x.AudioTags).FirstOrDefault(x => x.Id == id);
                dbSong.Title = song.Title;
                dbSong.Year = song.Year;
                dbSong.GenreId = song.GenreId;
                dbSong.ArtistId = song.ArtistId;
                dbSong.Duration = song.Duration;
                dbSong.Url = song.Url;

                _context.RemoveRange(dbSong.AudioTags);
                var audioTags = song.TagIds.Select(x => new AudioTag { AudioId = id, TagId = x });
                dbSong.AudioTags.AddRange(audioTags);

                _context.SaveChanges();

                return RedirectToAction(nameof(Index));
            }
            catch(Exception ex)
            {
                return View();
            }
        }

        // GET: SongController/Delete/5
        public ActionResult Delete(int id)
        {
            var song = _context.Audios
                .Include(x => x.Artist)
                .Include(x => x.Genre)
                .FirstOrDefault(x => x.Id == id);
            var songVM = new SongVM
            {
                Id = song.Id,
                Title = song.Title,
                Year = song.Year,
                GenreId = song.GenreId,
                GenreName = song.Genre.Name,
                ArtistId = song.ArtistId,
                ArtistName = song.Artist.Name,
                Duration = song.Duration,
                Url = song.Url
            };

            return View(songVM);
        }

        // POST: SongController/Delete/5
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult Delete(int id, Audio audio)
        {
            try
            {
                var dbAudio = _context.Audios.FirstOrDefault(x => x.Id == id);

                _context.Audios.Remove(dbAudio);

                _context.SaveChanges();

                return RedirectToAction(nameof(Index));
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }
    }
}
