﻿using exercise_6_7.Dtos;
using exercise_6_7.Models;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using Microsoft.EntityFrameworkCore;

// For more information on enabling Web API for empty projects, visit https://go.microsoft.com/fwlink/?LinkID=397860

namespace exercise_6_7.Controllers
{
    [Authorize(Roles = "Admin")]
    [Route("api/[controller]")]
    [ApiController]
    public class ReceiptController : ControllerBase
    {
        private readonly IConfiguration _configuration;
        private readonly Exercise6Context _context;

        public ReceiptController(IConfiguration configuration, Exercise6Context context)
        {
            _configuration = configuration;
            _context = context;
        }

        [HttpGet]
        public ActionResult<IEnumerable<ReceiptDto>> Get()
        {
            try
            {
                var receipts = _context.Receipts.Include(x => x.ReceiptItems);
                var result = MapToDto(receipts);

                return Ok(result);
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        // GET api/<ReceiptController>/5
        [HttpGet("{id}")]
        public ActionResult<ReceiptDto> Get(int id)
        {
            try
            {
                var receipt = _context.Receipts.Include(x => x.ReceiptItems).FirstOrDefault(x => x.Id == id);
                if (receipt == null)
                    return NotFound();

                var result = MapToDto(receipt);

                return Ok(result);
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        // POST api/<ReceiptController>
        [HttpPost]
        public ActionResult<ReceiptDto> Post([FromBody] ReceiptDto value)
        {
            try
            {
                var receipt = MapFromDto(value);
                _context.Receipts.Add(receipt);
                _context.SaveChanges();

                var result = MapToDto(receipt);
                return Ok(result);
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        // PUT api/<ReceiptController>/5
        [HttpPut("{id}")]
        public ActionResult<ReceiptDto> Put(int id, [FromBody] ReceiptDto value)
        {
            try
            {
                var receipt = _context.Receipts.Include(x => x.ReceiptItems).FirstOrDefault(x => x.Id == id);
                if (receipt == null)
                    return NotFound();

                receipt.Code = value.Code;
                receipt.Total = value.Total;
                receipt.IssuedAt = value.IssuedAt;

                // Easy way to update items: extract items from DTO and replace them
                var mappedReceipt = MapFromDto(value);
                _context.ReceiptItems.RemoveRange(receipt.ReceiptItems); // remove existing
                receipt.ReceiptItems = mappedReceipt.ReceiptItems; // add new

                // Easy way to update total: also from DTO
                receipt.Total = mappedReceipt.Total;

                _context.SaveChanges();

                var result = MapToDto(receipt);
                return Ok(result);
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        // DELETE api/<ReceiptController>/5
        [HttpDelete("{id}")]
        public ActionResult<ReceiptDto> Delete(int id)
        {
            try
            {
                var receipt = _context.Receipts.Include(x => x.ReceiptItems).FirstOrDefault(x => x.Id == id);
                if (receipt == null)
                    return NotFound();

                _context.ReceiptItems.RemoveRange(receipt.ReceiptItems); // First remove "child" items
                _context.Receipts.Remove(receipt); // Then remove "parent"
                _context.SaveChanges();

                var result = MapToDto(receipt);
                return Ok(result);
            }
            catch (Exception ex)
            {
                return StatusCode(500, ex.Message);
            }
        }

        private IEnumerable<ReceiptDto> MapToDto(IEnumerable<Receipt> receipts) =>
            receipts.Select(x => MapToDto(x));

        private ReceiptDto MapToDto(Receipt receipt)
        {
            var mappedItems = receipt.ReceiptItems.Select(item => new ReceiptItemDto
            {
                Id = item.Id,
                Price = item.Price,
                ProductId = item.ProductId,
                Quantity = item.Quantity,
                Value = item.Value
            }).ToList();

            var mappedReceipt = new ReceiptDto
            {
                Id = receipt.Id,
                Code = receipt.Code,
                Total = receipt.Total,
                IssuedAt = receipt.IssuedAt,
                ReceiptItems = mappedItems
            };

            return mappedReceipt;
        }

        private Receipt MapFromDto(ReceiptDto receipt) 
        {
            var mappedItems = receipt.ReceiptItems.Select(item => {
                var price = _context.Products.First(x => x.Id == item.ProductId).Price;

                return new ReceiptItem
                {
                    Id = item.Id,
                    ProductId = item.ProductId,
                    Price = price,
                    Quantity = item.Quantity,
                    Value = price * item.Quantity,
                };
            }).ToList();

            var mappedReceipt = new Receipt
            {
                Id = receipt.Id,
                Code = receipt.Code,
                Total = mappedItems.Sum(x => x.Value),
                IssuedAt = receipt.IssuedAt,
                ReceiptItems = mappedItems,
            };

            return mappedReceipt;
        }
            

    }
}
