﻿using exercise_3_4.Repository;
using exercise_3_8.Models;
using Microsoft.AspNetCore.Mvc;

// For more information on enabling Web API for empty projects, visit https://go.microsoft.com/fwlink/?LinkID=397860

namespace exercise_3_8.Controllers
{
    [Route("api/[controller]")]
    [ApiController]
    public class ProductsController : ControllerBase
    {
        private static void Log(int level, string message)
        {
            LogsRepository.LogItems.Add(
                new LogItem
                {
                    Timestamp = DateTime.Now,
                    Level = level,
                    Message = message
                });
        }

        [HttpGet]
        public ActionResult<IEnumerable<Product>> Get()
        {
            try
            {
                return Ok(ProductsRepository.Products);
            }
            catch (Exception ex)
            {
                Log(5, $"Error while retrieving all products: {ex.Message}");
                return StatusCode(500);
            }
        }

        [HttpGet("{id}")]
        public ActionResult<Product> Get(int id)
        {
            try
            {
                var product = ProductsRepository.Products.FirstOrDefault(x => x.Id == id);
                if (product == null)
                {
                    Log(0, $"Product {product.Id} not found");
                    return NotFound($"Could not find product with id {id}");
                }

                return Ok(product);
            }
            catch (Exception ex)
            {
                Log(5, $"Error while retrieving product {id}: {ex.Message}");
                return StatusCode(500);
            }
        }

        [HttpGet("[action]")]
        public ActionResult<IEnumerable<Product>> Search(string text, int start = 0, int count = 10)
        {
            try
            {
                IEnumerable<Product> results = ProductsRepository.Products;

                // Filtering
                if (!String.IsNullOrEmpty(text))
                {
                    // We can search both name and description for a given text
                    results = results.Where(s =>
                        s.Name.Contains(text, StringComparison.OrdinalIgnoreCase) ||
                        s.Description.Contains(text, StringComparison.OrdinalIgnoreCase));
                }

                // Paging
                results = results.Skip(start * count).Take(count);

                return Ok(results);
            }
            catch (Exception ex)
            {
                Log(5, $"Error while searching for products: {ex.Message}");
                return StatusCode(500);
            }
        }

        [HttpPost]
        public ActionResult<Product> Post([FromBody]Product product)
        {
            try
            {
                int maxId;
                if (ProductsRepository.Products.Any())
                {
                    maxId = ProductsRepository.Products.Max(x => x.Id);
                }
                else
                {
                    maxId = 0;
                }

                product.Id = maxId + 1;
                ProductsRepository.Products.Add(product);

                Log(0, $"Product {product.Id} added");

                return Ok(product);
            }
            catch (Exception ex)
            {
                Log(5, $"Error while creating product: {ex.Message}");
                return StatusCode(500);
            }
        }

        [HttpPut("{id}")]
        public ActionResult<Product> Put(int id, [FromBody] Product value)
        {
            try
            {
                var product = ProductsRepository.Products.FirstOrDefault(x => x.Id == id);
                if (product == null)
                {
                    Log(0, $"Product {product.Id} not found");
                    return NotFound($"Could not find product with id {id}");
                }

                product.Name = value.Name;
                product.Description = value.Description;
                product.CreationDate = value.CreationDate;
                product.Rating = value.Rating;

                Log(0, $"Product {id} updated");

                return Ok(product);
            }
            catch (Exception ex)
            {
                Log(5, $"Error while updating product {id}: {ex.Message}");
                return StatusCode(500);
            }
        }

        [HttpDelete("{id}")]
        public ActionResult<Product> Delete(int id)
        {
            try
            {
                var product = ProductsRepository.Products.FirstOrDefault(x => x.Id == id);
                if (product == null)
                {
                    Log(0, $"Product {product.Id} not found");
                    return NotFound($"Could not find product with id {id}");
                }

                ProductsRepository.Products.Remove(product);
                Log(0, $"Product {id} removed");

                return Ok(product);
            }
            catch (Exception ex)
            {
                Log(5, $"Error while removing product: {ex.Message}");
                return StatusCode(500);
            }
        }
    }
}
